//////////////////////////////////////////////////////////////////////////////
//  Copyright 2016 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals exec, registerCallback*/

/*!
 * ntp.acad - Interaction with AutoCAD
 */
 
// Namespace
var ntp = ntp || {};

/**
 * Functions for invoking AutoCAD functionality
 */
ntp.acad = (function () {
    // Change it to 'true' to enable sync call, 
    // by default we use asyncExec which is async
    var defaultSync = false;
    
    /**
     * Entry point for invoking AutoCAD function
     * 
     * @param {string|object} invokeInfo The ACAD function invoke info
     * @param {boolean} syncCall Whether this call is synchronized
     * @param {function} onSuccess Success callback
     * @param {function} onError Failure callback
     */
    var invokeAcadFunction = function (invokeInfo, syncCall) {
        // Parameter normalization
        if (typeof invokeInfo === 'undefined') {
            return null;
        } else if (typeof invokeInfo === 'object') {
            invokeInfo = JSON.stringify(invokeInfo);
        }
        if (typeof syncCall === 'undefined') {
            syncCall = defaultSync;
        }
        
        var invocation = $.Deferred();
        
        // Invoke ACAD method
        if (syncCall) {
            if (typeof exec === 'undefined') { 
                invocation.reject();
                return invocation.promise();
            }		
            var retJSON = exec(invokeInfo);
            var result = $.parseJSON(retJSON);
            if (result.retCode === 0) {
                invocation.resolve(JSON.stringify(result.retValue));
            } else {
                invocation.reject(JSON.stringify(result.retValue));
            }
        } else {
            if (typeof execAsync === 'undefined') { 
                invocation.reject();
                return invocation.promise();                
            }
            execAsync(invokeInfo, function (data) {
                var result = $.parseJSON(data);
                invocation.resolve(JSON.stringify(result.retValue));
            }, function (data) {
                var result = $.parseJSON(data);
                invocation.reject(JSON.stringify(result.retValue));
            });
        }
        
        return invocation.promise();
    }; // invokeAcadFunction()
    
    // ntp.acad
    return {
        invokeCustomAction: function (action) {
            return invokeAcadFunction({
                functionName: 'ntp.invokeCustomAction',
                invokeAsCommand: false,
                functionParams: { 'action': action}
            }, false);
        },

        notifyShortcutKeyPressed: function (event) {
            var keyKind = 0
            if (event.ctrlKey) { 
                keyKind = 1; // ctrl key
            } else {
                keyKind = 2; // alt key
            }
            return invokeAcadFunction({
                functionName: 'ntp.notifyShortcutKeyPressed',
                invokeAsCommand: false,
                functionParams: { 'keyKind': keyKind, 'keyCode': event.which }
            });
        },

        createDrawingFile: function () {
            return invokeAcadFunction({
                functionName: 'ntp.createDrawingFile',
                invokeAsCommand: false
            });
        },

        createDrawingFileWithTemplate: function (templatePath) {
            return invokeAcadFunction({
                functionName: 'ntp.createDrawingFileWithTemplate',
                invokeAsCommand: false,
                functionParams: { 'templatePath': templatePath }
            });
        },
        
        createDrawingFileWithNoTemplate: function (imperial) {
            return invokeAcadFunction({
                functionName: 'ntp.createDrawingFileWithNoTemplate',
                invokeAsCommand: false,
                functionParams: { 'imperial': imperial }
            });
        },

        showLoginDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showLoginDialog',
                invokeAsCommand: false
            }); 
        },

        showOpenFileDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showOpenFileDialog',
                invokeAsCommand: false
            });
        },

        showOpenSheetSetDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showOpenSheetSetDialog',
                invokeAsCommand: false
            });
        },

        showNewSheetSetDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showNewSheetSetDialog',
                invokeAsCommand: false
            });
        },

        showOpenLocalSampleFileDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showOpenLocalSampleFileDialog',
                invokeAsCommand: false
            });
        },

        pinDocumentWithPath: function (fullPath, pinIt) {
            return invokeAcadFunction({
                functionName: 'ntp.pinDocumentWithPath',
                invokeAsCommand: false,
                functionParams: { 'fullPath': fullPath, 'pinIt': pinIt}
            });
        },

        openDocumentWithPath: function (fullPath) {
            return invokeAcadFunction({
                functionName: 'ntp.openDocumentWithPath',
                invokeAsCommand: false,
                functionParams: { 'fullPath': fullPath }
            });
        },

        loadTemplateFilesAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.loadTemplateFiles',
                invokeAsCommand: false
            });
        },
        
        getSignedRequestDataAsJSON: function (url) {
            return invokeAcadFunction({
                functionName: 'ntp.getSignedRequestData',
                invokeAsCommand: false,
                functionParams: { 'url': url}
            });
        },

        loadTemplateFilesFromSelectedFolder: function () {
            return invokeAcadFunction({
                functionName: 'ntp.loadTemplateFilesFromSelectedFolder',
                invokeAsCommand: false
            });
        },

        loadRecentFilesAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.loadRecentFiles',
                invokeAsCommand: false
            });
        },
        
        
        loadCurrentProjectAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.loadCurrentProject',
                invokeAsCommand: false
            });
        },
        
        getLastUsedTemplateAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.getLastUsedTemplate',
                invokeAsCommand: false
            }, true);
        },

        handleProjectClicked: function (projectPath) {
            return invokeAcadFunction({
                functionName: 'ntp.handleProjectClicked',
                invokeAsCommand: false,
                functionParams: { 'projectPath': projectPath }
            });
        },
        
        showProjectNavigator: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showProjectNavigator',
                invokeAsCommand: false
            });
        },
        
        newProject: function () {
            return invokeAcadFunction({
                functionName: 'ntp.newProject',
                invokeAsCommand: false
            });
        },
        
        sendFeedbackWithBrowser: function () {
            return invokeAcadFunction({
                functionName: 'ntp.sendFeedback',
                invokeAsCommand: false
            });
        },

        openUserHomePageWithBrowser: function () {
            return invokeAcadFunction({
                functionName: 'ntp.openUserHomePage',
                invokeAsCommand: false
            });
        },

        openUrlWithBrowser: function (url) {
            return invokeAcadFunction({
                functionName: 'ntp.openUrlWithBrowser',
                functionParams: { 'url': url },
                invokeAsCommand: false
            }, false);
        },

        openUrlWithHelpBrowser: function (topic_id) {
            return invokeAcadFunction({
                functionName: 'ntp.openUrlWithHelpBrowser',
                functionParams: { 'helpTopicId': topic_id },
                invokeAsCommand: false
            }, false);
        },

        getHardwareAccelerationInfoAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.getHardwareAccelerationInfo',
                invokeAsCommand: false
            });
        },
        
        getLicenseInfoAsJSON: function() {
            return invokeAcadFunction({
                functionName: 'ntp.getLicenseInfo',
                invokeAsCommand: false
            });
        },
        
        getUpdateInfoAsJSON: function() {
            return invokeAcadFunction({
                functionName: 'ntp.getUpdateInfo',
                invokeAsCommand: false
            });
        },
        
        getOfflineHelpInfoAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.getOfflineHelpInfo',
                invokeAsCommand: false
            });
        },
        
        getSignedUrlInfoAsJSON: function (url) {
            return invokeAcadFunction({
                functionName: 'ntp.getSignedURL',
                functionParams: { 'url': url },
                invokeAsCommand: false
            });
        },

        showProductUpdateDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showProductUpdateDialog',
                invokeAsCommand: false
            });
        },
        
        showGsConfigDialog: function () {
            return invokeAcadFunction({
                functionName: 'ntp.showGsConfigDialog',
                invokeAsCommand: false
            });
        },
        stopVideoById: function (id) {
            return invokeAcadFunction({
                functionName: 'ntp.stopVideoById',
                functionParams: { 'id': id },
                invokeAsCommand: false
            });
        },
        
        openDroppedFiles: function (id) {
            return invokeAcadFunction({
                functionName: 'ntp.openDroppedFiles',
                invokeAsCommand: false
            });
        },
                
        getUserSettingAsJSON: function (name) {
            return invokeAcadFunction({
                functionName: 'ntp.getUserSetting',
                functionParams: { 'name': name },
                invokeAsCommand: false
            });
        },
        
        updateUserSetting: function (name, value) {
            return invokeAcadFunction({
                functionName: 'ntp.updateUserSetting',
                functionParams: { 'name' : name, 'value' : value },
                invokeAsCommand: false
            });
        },

        updateLearnContentState: function (name, value) {
            return invokeAcadFunction({
                functionName: 'ntp.updateLearnContentState',
                functionParams: { 'name' : name, 'value' : value },
                invokeAsCommand: false
            });
        },

        getLearnContentStateListAsJSON: function (videoList) {
            return invokeAcadFunction({
                functionName: 'ntp.getLearnContentStateList',
                functionParams: { 'contentList': videoList },
                invokeAsCommand: false
            });
        },

        getProductInfoAsJSON: function () {
            return invokeAcadFunction({
                functionName: 'ntp.getProductInfo',
                invokeAsCommand: false
            });
        },

        openAlertLink: function (id, url) {
            return invokeAcadFunction({
                functionName: 'ntp.openAlertLink',
                functionParams: {'id': id, 'url': url},
                invokeAsCommand: false
            });
        },

        getAlertViewState: function (alerts) {
            return invokeAcadFunction({
                functionName: 'ntp.getAlertViewState',
                functionParams: {'alerts': alerts},
                invokeAsCommand: false
            });
        }
    };
})(); // ntp.acad

/**
 * Registers Javascript callbacks for AutoCAD
 */
$(document).ready(function () {    
    if (typeof registerCallback === 'undefined') {
        // registerCallback undefined. The HTML is not opened inside AutoCAD
        return;
    }
    
    // Register JS functions that can be called from C++ side.
    // These functions are defined in script.js
    registerCallback('ntp_continueWorking.refreshWithJSON',
        ntp.utils.makeCallback(ntp.widget.continueWorking, 'refreshWithJSON'));
    registerCallback('ntp_templatebar.refreshWithJSON',
        ntp.utils.makeCallback(ntp.widget.templatebar, 'refreshWithJSON'));
    registerCallback('ntp_templatebar.refreshLastUsedTemplateWithJSON',
        ntp.utils.makeCallback(ntp.widget.templatebar, 'refreshLastUsedTemplateWithJSON'));
    registerCallback('ntp_learnTab.unhilightRecentViewed',
        ntp.utils.makeCallback(ntp.content.learnTab, 'unhilightRecentViewed'));
    registerCallback('ntp_projectbar.refreshCurrentProjectWithJSON',
        ntp.utils.makeCallback(ntp.widget.projectbar, 'refreshCurrentProjectWithJSON'));
    registerCallback('ntp_projectbar.refreshWithJSON',
        ntp.utils.makeCallback(ntp.widget.projectbar, 'refreshWithJSON'));
    registerCallback('ntp_connect.refreshLoginWithJSON',
        ntp.utils.makeCallback(ntp.widget.connect, 'refreshLoginWithJSON'));
    registerCallback('ntp_notifications.refreshHwaccWithJSON',
        ntp.utils.makeCallback(ntp.widget.notifications, 'refreshHwaccWithJSON'));
    registerCallback('ntp_notifications.refreshUpdateWithJSON',
        ntp.utils.makeCallback(ntp.widget.notifications, 'refreshUpdateWithJSON'));
    registerCallback('ntp_notifications.refreshLicenseWithJSON',
        ntp.utils.makeCallback(ntp.widget.notifications, 'refreshLicenseWithJSON'));
    registerCallback('ntp_content.refreshTryTabWithJSON',
        ntp.utils.makeCallback(ntp.content.tryTab, 'refreshTryTabWithJSON'));
    registerCallback('ntp_video.stopWithJSON',
        ntp.utils.makeCallback(ntp.video, 'stopWithJSON'));
    registerCallback('ntp.unviewed_alerts', 
        ntp.utils.makeCallback(ntp.widget.notifications, 'refreshAlertsNumber'));
    registerCallback('ntp_widgets.bindToolTips',
        ntp.utils.makeCallback(ntp.widget, 'bindToolTips'));
    registerCallback('ntp_widgets.unbindToolTips',
        ntp.utils.makeCallback(ntp.widget, 'unbindToolTips'));
});

// SIG // Begin signature block
// SIG // MIIZ7QYJKoZIhvcNAQcCoIIZ3jCCGdoCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // k3L7DnMq8RPZic1U7OaqWsjJRk/IJEbKDIDTSl/YEbyg
// SIG // ggpjMIIFAjCCA+qgAwIBAgIQZLyxx87k2lhp1Y8eIvvp
// SIG // 9DANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQTAeFw0xNTA5MDMwMDAwMDBaFw0x
// SIG // NjA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwKU2FuIFJh
// SIG // ZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIEluYzEfMB0G
// SIG // A1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91cDEWMBQG
// SIG // A1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJKoZIhvcN
// SIG // AQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARqd9dP4fq2
// SIG // qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8WR/0tW17
// SIG // lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpRIG9QthGU
// SIG // q8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGVplDOyr97
// SIG // 6ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm+eRliYKe
// SIG // KF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82CeRBCToR1E
// SIG // 7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIzqVo3A0iM
// SIG // HBsbioxa5DYyXuEfNCktosUlFXfegd514ZJns6YDr4Pl
// SIG // ahC0+lK3hgJy5vkCAwEAAaOCAW4wggFqMAkGA1UdEwQC
// SIG // MAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMDMGYGA1UdIARfMF0wWwYLYIZIAYb4RQEHFwMw
// SIG // TDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNiLmNv
// SIG // bS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6Ly9kLnN5
// SIG // bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAUljtT8Hkzl699
// SIG // g+8uK8zKt4YecmYwKwYDVR0fBCQwIjAgoB6gHIYaaHR0
// SIG // cDovL3N2LnN5bWNiLmNvbS9zdi5jcmwwVwYIKwYBBQUH
// SIG // AQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc3Yuc3lt
// SIG // Y2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vc3Yuc3lt
// SIG // Y2IuY29tL3N2LmNydDARBglghkgBhvhCAQEEBAMCBBAw
// SIG // FgYKKwYBBAGCNwIBGwQIMAYBAQABAf8wDQYJKoZIhvcN
// SIG // AQELBQADggEBAB6BYdY8nzLWS3nsk/xM5CWdizR4wyDn
// SIG // 8wfGVbcuEq/88byNnY0ODxisa7AELvLe6b48DorFFP2n
// SIG // Hw0qOeIxfg2w+1DzWzYQhMfLi4V/Pp1OmCYvYEsTb/oU
// SIG // +YbiSgSFjrN5Pi2LN5Oxw5Ey7WAK2vhWr4EL/7QDHoso
// SIG // 4a3lS3WklPz2K/Nlhk4i8S230BlGCHsZGDY7PtfowWdz
// SIG // cq50xCEIScERSRtLd+YL6SWXk+6gX9HVcFxtvLQ4N9Df
// SIG // F/BKk13pYgyvZPkedIKA4yYCy4/VjouNoKhgM/aaGLAy
// SIG // Dv3ujRdXLCcosR4MbD8X1v9z6WkgG5YZZicWPPsJN7ty
// SIG // B1wwggVZMIIEQaADAgECAhA9eNf5dklgsmF99PAeyoYq
// SIG // MA0GCSqGSIb3DQEBCwUAMIHKMQswCQYDVQQGEwJVUzEX
// SIG // MBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsT
// SIG // FlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOjA4BgNVBAsT
// SIG // MShjKSAyMDA2IFZlcmlTaWduLCBJbmMuIC0gRm9yIGF1
// SIG // dGhvcml6ZWQgdXNlIG9ubHkxRTBDBgNVBAMTPFZlcmlT
// SIG // aWduIENsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlm
// SIG // aWNhdGlvbiBBdXRob3JpdHkgLSBHNTAeFw0xMzEyMTAw
// SIG // MDAwMDBaFw0yMzEyMDkyMzU5NTlaMH8xCzAJBgNVBAYT
// SIG // AlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlv
// SIG // bjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29y
// SIG // azEwMC4GA1UEAxMnU3ltYW50ZWMgQ2xhc3MgMyBTSEEy
// SIG // NTYgQ29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAl4MeABavLLHSCMTXaJNR
// SIG // YB5x9uJHtNtYTSNiarS/WhtR96MNGHdou9g2qy8hUNqe
// SIG // 8+dfJ04LwpfICXCTqdpcDU6kDZGgtOwUzpFyVC7Oo9tE
// SIG // 6VIbP0E8ykrkqsDoOatTzCHQzM9/m+bCzFhqghXuPTbP
// SIG // HMWXBySO8Xu+MS09bty1mUKfS2GVXxxw7hd924vlYYl4
// SIG // x2gbrxF4GpiuxFVHU9mzMtahDkZAxZeSitFTp5lbhTVX
// SIG // 0+qTYmEgCscwdyQRTWKDtrp7aIIx7mXK3/nVjbI13Iwr
// SIG // b2pyXGCEnPIMlF7AVlIASMzT+KV93i/XE+Q4qITVRrgT
// SIG // hsIbnepaON2b2wIDAQABo4IBgzCCAX8wLwYIKwYBBQUH
// SIG // AQEEIzAhMB8GCCsGAQUFBzABhhNodHRwOi8vczIuc3lt
// SIG // Y2IuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwbAYDVR0g
// SIG // BGUwYzBhBgtghkgBhvhFAQcXAzBSMCYGCCsGAQUFBwIB
// SIG // FhpodHRwOi8vd3d3LnN5bWF1dGguY29tL2NwczAoBggr
// SIG // BgEFBQcCAjAcGhpodHRwOi8vd3d3LnN5bWF1dGguY29t
// SIG // L3JwYTAwBgNVHR8EKTAnMCWgI6Ahhh9odHRwOi8vczEu
// SIG // c3ltY2IuY29tL3BjYTMtZzUuY3JsMB0GA1UdJQQWMBQG
// SIG // CCsGAQUFBwMCBggrBgEFBQcDAzAOBgNVHQ8BAf8EBAMC
// SIG // AQYwKQYDVR0RBCIwIKQeMBwxGjAYBgNVBAMTEVN5bWFu
// SIG // dGVjUEtJLTEtNTY3MB0GA1UdDgQWBBSWO1PweTOXr32D
// SIG // 7y4rzMq3hh5yZjAfBgNVHSMEGDAWgBR/02Wnwt3su/Aw
// SIG // CfNDOfoCrzMxMzANBgkqhkiG9w0BAQsFAAOCAQEAE4Ua
// SIG // HmmpN/egvaSvfh1hU/6djF4MpnUeeBcj3f3sGgNVOftx
// SIG // lcdlWqeOMNJEWmHbcG/aIQXCLnO6SfHRk/5dyc1eA+CJ
// SIG // nj90Htf3OIup1s+7NS8zWKiSVtHITTuC5nmEFvwosLFH
// SIG // 8x2iPu6H2aZ/pFalP62ELinefLyoqqM9BAHqupOiDlAi
// SIG // KRdMh+Q6EV/WpCWJmwVrL7TJAUwnewusGQUioGAVP9rJ
// SIG // +01Mj/tyZ3f9J5THujUOiEn+jf0or0oSvQ2zlwXeRAwV
// SIG // +jYrA9zBUAHxoRFdFOXivSdLVL4rhF4PpsN0BQrvl8OJ
// SIG // IrEfd/O9zUPU8UypP7WLhK9k8tAUITGCDuIwgg7eAgEB
// SIG // MIGTMH8xCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1h
// SIG // bnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50
// SIG // ZWMgVHJ1c3QgTmV0d29yazEwMC4GA1UEAxMnU3ltYW50
// SIG // ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
// SIG // AhBkvLHHzuTaWGnVjx4i++n0MA0GCWCGSAFlAwQCAQUA
// SIG // oHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
// SIG // MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwG
// SIG // CisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIGUmylEl
// SIG // 8N+H0agPWW5ocq9HCB2Ui+h0TM6CMwk6+Tu6MA0GCSqG
// SIG // SIb3DQEBAQUABIIBAGyPReOvbX7D34gV24pWTX4+tJhg
// SIG // 1uVZMoipQj5s+Qy8Fe6Tis2vI1ZfxF8G3l5Wh97IP/qX
// SIG // LHl+N9Uorvo8nbuDlf5r1WHfauCJK37+QejFdV3JT4Wl
// SIG // 2ZFp1F4jY3o/GywDmeImjtlB3pjxXWDgMv/bk5I52PsF
// SIG // Q1DMeClH2bTeMWpjs7hXMgn2+mabffnmM2+B7U+0ANQK
// SIG // MvlCt3/LT8rCUxLoV3e42qyREyduL8sRNgLqYFeJv2t7
// SIG // mt7pVGxuSUMRDaw9AymXzZEWa5FhHmZSga99BjocdNLx
// SIG // 7AzYr0YpYA9GM9yseW7PTkji/3IEkbXPwKo2G4cjQcGa
// SIG // J/njm/WhggyhMIIMnQYKKwYBBAGCNwMDATGCDI0wggyJ
// SIG // BgkqhkiG9w0BBwKgggx6MIIMdgIBAzEPMA0GCWCGSAFl
// SIG // AwQCAQUAMIHdBgsqhkiG9w0BCRABBKCBzQSByjCBxwIB
// SIG // AQYJKwYBBAGgMgIDMDEwDQYJYIZIAWUDBAIBBQAEIOAd
// SIG // iPIuZzcj5NM2h/TJqXVUhO5rs5OCNMKaMk/C+BzoAhQZ
// SIG // aoIoWteGBxEtj8QzagFfSBPTiRgPMjAxNjAyMDcwNDM3
// SIG // NDBaoF2kWzBZMQswCQYDVQQGEwJTRzEfMB0GA1UEChMW
// SIG // R01PIEdsb2JhbFNpZ24gUHRlIEx0ZDEpMCcGA1UEAxMg
// SIG // R2xvYmFsU2lnbiBUU0EgZm9yIEFkdmFuY2VkIC0gRzKg
// SIG // ggjGMIIEqTCCA5GgAwIBAgISESEWwAmY3MaPon0lw4Y2
// SIG // qIO7MA0GCSqGSIb3DQEBCwUAMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyMB4XDTE1MDIwMzAwMDAwMFoXDTI2
// SIG // MDMwMzAwMDAwMFowWTELMAkGA1UEBhMCU0cxHzAdBgNV
// SIG // BAoTFkdNTyBHbG9iYWxTaWduIFB0ZSBMdGQxKTAnBgNV
// SIG // BAMTIEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAt
// SIG // IEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEAt8eSiijY85HLUA1d6ylEwYJZpcpkzr2DtVHllMxu
// SIG // gJoXbNz/YuY2Q14Bwgxkjcp3nvZdd0W5Xsgk60E3EWY3
// SIG // rMjNdJvMs+9M4n8bznO+yZadNR5891T3bOygEtaumZ1j
// SIG // xHhfWakuIvXDJ9iyW0VjJ5D/jLE9qbTzDsDxIGUGiZ0D
// SIG // HKOwUkGFNOu8y85U9hL+swGJSqb6VMUf+9McwOO/l+M/
// SIG // 0Bv5QMcb1esmfEMwx3DyeBGqhU6lDg0ASjoRd6PKIwGA
// SIG // nsiTVl1GzIrY0qoKhIWyiVvVMs5BexhOVMqr0VJeEtMk
// SIG // nlbzDI/dvZKOedVBicOjoOWR/MdnnYczznEycwIDAQAB
// SIG // o4IBZzCCAWMwDgYDVR0PAQH/BAQDAgeAMEwGA1UdIARF
// SIG // MEMwQQYJKwYBBAGgMgEeMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMAkGA1UdEwQCMAAwFgYDVR0lAQH/BAwwCgYIKwYB
// SIG // BQUHAwgwRgYDVR0fBD8wPTA7oDmgN4Y1aHR0cDovL2Ny
// SIG // bC5nbG9iYWxzaWduLmNvbS9ncy9nc3RpbWVzdGFtcGlu
// SIG // Z3NoYTJnMi5jcmwwWAYIKwYBBQUHAQEETDBKMEgGCCsG
// SIG // AQUFBzAChjxodHRwOi8vc2VjdXJlLmdsb2JhbHNpZ24u
// SIG // Y29tL2NhY2VydC9nc3RpbWVzdGFtcGluZ3NoYTJnMi5j
// SIG // cnQwHQYDVR0OBBYEFC1ubtGN5QOA7udj6afZ2gs8VyI9
// SIG // MB8GA1UdIwQYMBaAFJIhp0qVXWSwm7Qe5gA3R+adQStM
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQB/CgOhVXb+qvoU0RF6
// SIG // 4uIPOjL/e0bGBHGMXtDMeUfddQdtuki+P2DiyxEgwrar
// SIG // Q8e3otm4Wv7NtDqz8VJTDLSO/2LdiJuyGn8bUbdxh9pD
// SIG // SzdSzf4+BilwpCds59w8f3bjkMga6ZkSUO7qTCMayF+c
// SIG // 1SmEKuaBYgZ2gnajE1UOSKZR0fNMDTwIICiGaoO3nQyg
// SIG // XmA0BjGofXGPUfIE36Imk7nVKHejoLdCo4swJ1oJ9tyD
// SIG // mCHVfUGaBxkUWih2fa2goyQBTuEWyL9/R/M6BF5lPqJM
// SIG // //1vBs+suBsAZ/rva37X4gq/e8WnEqzXNPVXiD6Elq1r
// SIG // VC49Jw/iORrBxxhyMIIEFTCCAv2gAwIBAgILBAAAAAAB
// SIG // MYnGUAQwDQYJKoZIhvcNAQELBQAwTDEgMB4GA1UECxMX
// SIG // R2xvYmFsU2lnbiBSb290IENBIC0gUjMxEzARBgNVBAoT
// SIG // Ckdsb2JhbFNpZ24xEzARBgNVBAMTCkdsb2JhbFNpZ24w
// SIG // HhcNMTEwODAyMTAwMDAwWhcNMjkwMzI5MTAwMDAwWjBb
// SIG // MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2ln
// SIG // biBudi1zYTExMC8GA1UEAxMoR2xvYmFsU2lnbiBUaW1l
// SIG // c3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAKqbjsOrEVEl
// SIG // AbaWlOJP2MEI9kYj2UXFlZdbqxq/0mxXyTMGH6APxjx+
// SIG // U0h6v52Hnq/uw4xH4ULs4+OhSmwMF8SmwbnNW/EeRImO
// SIG // /gveIVgT7k3IxWcLHLKz8TR2kaLLB203xaBHJgIVpJCR
// SIG // qXme1+tXnSt8ItgU1/EHHngiNmt3ea+v+X+OTuG1CDH9
// SIG // 6u1LcWKMI/EDOY9EebZ2A1eerS8IRtzSjLz0jnTOyGhp
// SIG // UXYRiw9dJFsZVD0mzECNgicbWSB9WfaTgI74Kjj9a6BA
// SIG // ZR9XdsxbjgRPLKjbhFATT8bci7n43WlMiOucezAm/HpY
// SIG // u1m8FHKSgVe3dsnYgAqAbgkCAwEAAaOB6DCB5TAOBgNV
// SIG // HQ8BAf8EBAMCAQYwEgYDVR0TAQH/BAgwBgEB/wIBADAd
// SIG // BgNVHQ4EFgQUkiGnSpVdZLCbtB7mADdH5p1BK0wwRwYD
// SIG // VR0gBEAwPjA8BgRVHSAAMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9jcmwu
// SIG // Z2xvYmFsc2lnbi5uZXQvcm9vdC1yMy5jcmwwHwYDVR0j
// SIG // BBgwFoAUj/BLf6guRSSuTVD6Y5qL3uLdG7wwDQYJKoZI
// SIG // hvcNAQELBQADggEBAARWgkp80M7JvzZm0b41npNsl+gG
// SIG // zjEYWflsQV+ALsBCJbgYx/zUsTfEaKDPKGoDdEtjl4V3
// SIG // YTvXL+P1vTOikn0RH56KbO8ssPRijTZz0RY28bxe7LSA
// SIG // mHj80nZ56OEhlOAfxKLhqmfbs5xz5UAizznO2+Z3lae7
// SIG // ssv2GYadn8jUmAWycW9Oda7xPWRqO15ORqYqXQiS8aPz
// SIG // HXS/Yg0jjFwqOJXSwNXNz4jaHyi1uoFpZCq1pqLVc6/c
// SIG // RtsErpHXbsWYutRHxFZ0gEd4WIy+7yv97Gy/0ZT3v1Dg
// SIG // e+CQ/SAYeBgiXQgujBygl/MdmX2jnZHTBkROBG56HCDj
// SIG // NvC2ULkxggK0MIICsAIBATBxMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyAhIRIRbACZjcxo+ifSXDhjaog7sw
// SIG // DQYJYIZIAWUDBAIBBQCgggEUMBoGCSqGSIb3DQEJAzEN
// SIG // BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMTYw
// SIG // MjA3MDQzNzQwWjAvBgkqhkiG9w0BCQQxIgQg+92Rxaci
// SIG // HLRfbPJEbBP46NPvdAQoknEnRXuxtg0xKqAwgaYGCyqG
// SIG // SIb3DQEJEAIMMYGWMIGTMIGQMIGNBBRXqt6jTjqEJxGX
// SIG // sll4jXMMauIuyTB1MF+kXTBbMQswCQYDVQQGEwJCRTEZ
// SIG // MBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UE
// SIG // AxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBT
// SIG // SEEyNTYgLSBHMgISESEWwAmY3MaPon0lw4Y2qIO7MA0G
// SIG // CSqGSIb3DQEBAQUABIIBAIlVkQLgM2OczIn/m8JldFC8
// SIG // 4vFj7Oz2VWFqPCAOcniC6N1XSz6Vg8D1IDVjmv8FW/Pt
// SIG // q6hjSyGDtzF99L21b1MzbjSvFbco0AB4fB3ASMdEkdnf
// SIG // BzGuuVt327Ic/UFU17/IHS5vmTOxjNC/NmuV4hP6RfKU
// SIG // FwnirA+O6rpPPAJbSyLZVMxtRKBp7T3Ou0f01V2Pj1Mg
// SIG // AGj3rhb0UgkZ4697JfAmdf8PGDt7cp9Wr9duIKHJ0OwO
// SIG // 7aIvw2+S1chHHQTeMKj9RSayp3SGak0LyrE3VYDs4jap
// SIG // g9irvU83EwTXITuuKVxnL7IL8jxEz4CKRCUgdkW9Trov
// SIG // ItwMJMuZY9M=
// SIG // End signature block
